type THeuristic<T> = (node: T, previousScore: number) => number;
type TNodeService<T, U> = (nodeValue: T) => Promise<Array<U>>;
type TStepResolver<T, U> = (item: T) => U | null;

interface IScore {
    /**
     * Current score of the current node
     */
    score: number;
}

/**
 * Class to manage cluster of path.
 */
class Cluster<T> {
    private nodes: Array<{ node: T } & IScore>;

    public constructor() {
        this.nodes = [];
    }

    /**
     * Add new Node to Cluster
     * @param node New path-node
     */
    public push(node: { node: T } & IScore) {
        this.nodes.push(node);
        return this;
    }

    /**
     * Remove the latest Node
     */
    public pop() {
        this.nodes.pop();
        return this;
    }

    /**
     * Create and copy all nodes to a new Cluster
     */
    public clone() {
        const cluster = new Cluster<T>();
        cluster.nodes = this.nodes.slice();
        return cluster;
    }

    /**
     * Return global score of Cluster
     */
    public getScore() {
        return this.nodes.reduce((prev, current) => prev + current.score, 0);
    }

    /**
     * Reset Cluster
     */
    public clear() {
        this.nodes.splice(0, this.nodes.length);
    }

    /**
     * Get full path
     */
    public getPath() {
        return this.nodes.map(node => node.node);
    }
}

/**
 * Base class of A* pathfinding
 */
export class AStar<TNode, TNodeValue> {
    private heuristics: Map<string, Array<THeuristic<TNode>>>;
    private nodeService: TNodeService<TNodeValue, TNode>;

    private stepResolver: TStepResolver<TNode, TNodeValue>;

    private selectedCluster: Cluster<TNode> | null;
    private currentCluster: Cluster<TNode>;

    private nodeMap: Set<TNodeValue>;

    constructor() {
        this.heuristics = new Map();
        this.nodeService = _ => Promise.resolve([]);

        this.selectedCluster = null;
        this.currentCluster = new Cluster();

        this.nodeMap = new Set();

        this.stepResolver = (item: TNode) => null;
    }

    /**
     * Add a new heuristic to a group
     * @param groupName Group of heuristic
     * @param heuristic Heuristic algo
     */
    public addHeuristics(groupName: string, heuristic: THeuristic<TNode>) {
        if (!this.heuristics.has(groupName)) {
            this.heuristics.set(groupName, []);
        }

        this.heuristics.get(groupName)!.push(heuristic);

        return this;
    }

    /**
     * Set Node service to retrieve all Neightboor's nodes
     * @param nodeService Service callback
     */
    public setNodeService(nodeService: TNodeService<TNodeValue, TNode>) {
        this.nodeService = nodeService;
        return this;
    }

    /**
     * Set resolver to get the attribute-field to explore new step
     * @param stepResolver Callback to return next step's field
     */
    public setStepResolver(stepResolver: TStepResolver<TNode, TNodeValue>) {
        this.stepResolver = stepResolver;
        return this;
    }

    /**
     * Launch pathfinding
     * @param from Start point
     * @param to End point
     * @param heuristicName Heuristic's group name
     */
    public find(from: TNodeValue, to: TNodeValue, heuristicName: string): Promise<Cluster<TNode> | null> {
        this.nodeMap.clear();

        this.currentCluster.clear();
        this.selectedCluster = null;

        return this._find(from, to, heuristicName);
    }

    private async _find(from: TNodeValue, to: TNodeValue, heuristicName: string) {

        if (from === to) {
            if (!this.selectedCluster || this.selectedCluster.getScore() > this.currentCluster.getScore()) {
                this.selectedCluster = this.currentCluster.clone();
                return this.selectedCluster;
            }
        }

        const heuristics = this.heuristics.get(heuristicName)!;

        this.nodeMap.add(from);

        const nodes = await this.nodeService(from);
        const nodesScore = nodes.map(node => {
            return {
                node,
                score: heuristics.reduce((prev, curr) => curr(node, prev), 0)
            };
        }).sort((a, b) => a.score - b.score);

        for (let i = 0, max = nodesScore.length; i < max; ++i) {
            const node = nodesScore[i];
            if (!this.selectedCluster || this.selectedCluster.getScore() > this.currentCluster.getScore() + node.score) {

                const nextStep = this.stepResolver(node.node);
                if (!!nextStep && !this.nodeMap.has(nextStep)) {
                    this.currentCluster.push(node);
                    await this._find(nextStep, to, heuristicName);
                    this.currentCluster.pop();
                }
            }
        }

        this.nodeMap.delete(from);

        return this.selectedCluster;
    }
}
