import { ICommand, ICommandResponse } from './ICommand';
import CitiesDb from '../resources/cities';
import { ICity, ICityOrdered } from '../../shared/models/cities';
import { cache } from '../../shared/@decorators/function';

interface ITokens {
    /**
     * List of tokens
     */
    tokens: Array<string>;
    /**
     * List of score by token
     */
    scoring: Array<number>;
}

/**
 * Fetch max score of the current part
 * @param originalLength Full content length
 * @param currentLength Current content length
 * @param maxDepth Max depth for recursion
 * @param depth Current depth
 */
function getScore(originalLength: number, currentLength: number, maxDepth: number, depth: number) {
    return originalLength - (originalLength * depth / maxDepth) + (originalLength === currentLength ? originalLength : originalLength - currentLength) * (maxDepth - depth);
}

/**
 * Tokenize input
 * We want to keep at least the first letter
 * @example
 * Input: Paris
 * paris|
 * p.?ris|p.?.?is|p.?r.?s|p.?ri.?|
 * pa.?is|pa.?.?s|pa.?i.?|
 * par.?s|par.?.?|
 * pari.?
 * @param content Current content to tokenize
 * @param maxDepth Max depth for recursion
 * @param keepPrefixLength How many letter we want to keep at the beginning
 * @param depth Current depth
 * @param originalContent Original full content
 */
function tokenize(content: string, maxDepth: number, keepPrefixLength: number, depth: number = 0, originalContent: string = content): ITokens {
    const maxScoring = getScore(originalContent.length, content.length, maxDepth, depth);
    if (depth >= maxDepth) {
        return { tokens: [content], scoring: [maxScoring] };
    }
    const tokens: Array<string> = [];
    const scoring: Array<number> = [];
    for (let i = 0, max = content.length; i <= max; ++i) {
        if (!i) {
            tokens.push(content);
            scoring.push(maxScoring);
        } else if (depth > 0 || i > keepPrefixLength) {
            const prefix = content.substr(0, i - 1);
            const subTokenize = tokenize(content.substr(i), maxDepth, Math.max(keepPrefixLength - i, 0), depth + 1, originalContent);
            subTokenize.tokens.forEach(suffix => tokens.push(`${prefix}.?${suffix}`));

            const partScoring = 1 / content.length * i;
            subTokenize.scoring.forEach((score, index) => scoring.push(score + partScoring * index));
        }
    }

    return { tokens, scoring };
}

/**
 * Apply matching-score for each item
 * @param cities List of city fetched
 * @param tokenized List of tokens
 */
function applyScoring(cities: Set<ICity>, tokenized: ITokens) {
    const regExpList = tokenized.tokens.map(stringRegExp => new RegExp(stringRegExp, 'ig'));
    const result = new Set<ICityOrdered>();
    cities.forEach(city => {
        const score = regExpList.reduce((score, regexp, index) => {
            let match: RegExpExecArray | null;
            while ((match = regexp.exec(city.name)) !== null) {
                score += tokenized.scoring[index] * ((city.name.length - match.index) / city.name.length);
            }
            return score;
        }, 0);
        result.add(Object.assign({
            score
        }, city));
    });
    return result;
}

interface IAutocompleteResponse extends ICommandResponse {

}

/**
 * Autocomplete command to retrieve list of City
 */
export class Autocomplete implements ICommand<IAutocompleteResponse> {
    public constructor(
        private fuzziness: number = 2,
        private keepPrefixLength: number = 1
    ) { }

    /**
     * Get the list of City from Full/Partial city name
     * @param cityName Full/Partial City name
     */
    @cache(30)
    public execute(cityName: string) {
        return new Promise((resolve, reject) => {

            const tokenized = tokenize(cityName, this.fuzziness, this.keepPrefixLength);
            const regExp = new RegExp(`(${tokenized.tokens.join('|')})`, 'i');
            CitiesDb.find(regExp)
                .then(result => !cityName ? result : applyScoring(result, tokenized))
                .then(resolve)
                .catch(reject);
        });
    }
}
