import { IAdapter } from './IAdapter';

type TWorkerExecuteMethod<T> = (command: string, args?: Array<string>) => T;

/**
 * Adapter for service WebWorker
 */
export class WebWorkerAdapter implements IAdapter {
    private static instance: Promise<TWorkerExecuteMethod<any>>;

    private static launchService() {
        if (WebWorkerAdapter.instance) {
            return;
        }

        WebWorkerAdapter.instance = new Promise((resolve, reject) => {

            /**
             * Initialize WebWorker service
             * @param dbWorker WebWorker instance
             */
            const initializeWorker = (dbWorker: Worker) => {
                const requestCache: {
                    [id: string]: [Function, Function]
                } = {};
                let requestId = 0;

                dbWorker.onerror = (e) => {
                    console.error(e);
                    Object.keys(requestCache).forEach(key => {
                        const [_, reject] = requestCache[key];
                        delete requestCache[key];
                        reject('Error service');
                    });
                };

                dbWorker.onmessage = (e) => {
                    const { id, response } = e.data;
                    const [resolve] = requestCache[id];
                    delete requestCache[id];
                    resolve(response);
                };

                return (command: string, args: Array<string> = []): Promise<TWorkerExecuteMethod<any>> => {
                    return new Promise((resolve, reject) => {
                        const id = requestId++;

                        requestCache[id] = [resolve, reject];
                        dbWorker.postMessage({
                            id,
                            command,
                            args
                        });
                    });
                };
            };

            try {
                // Load WebWorker
                const dbWorker = new Worker('./searchengine/build.js');
                resolve(initializeWorker(dbWorker));
            } catch (e) {
                // Fallback for security issue (like file:/// protocole)

                const script = document.createElement('script');
                script.onload = () => {
                    // @ts-ignore Workaround for Chrome file protocole
                    const dbWorker = new Worker(URL.createObjectURL(new Blob(['(' + worker_main.toString() + ')()'], { type: 'text/javascript' })));
                    resolve(initializeWorker(dbWorker));
                };
                script.onerror = (e) => {
                    console.error(e);
                    reject('Error service');
                };
                script.src = './searchengine/build.js';
                document.querySelector('head')!.appendChild(script);
            }
        });
    }

    public constructor(private minDelay: number = 10) {
        WebWorkerAdapter.launchService();
    }

    fetch<T>(method: string, args: string[]): Promise<T> {
        return Promise.all([
            WebWorkerAdapter.instance.then(execute => execute(method, args)), // Real service request
            this.sleep() // Simulate latency
        ]).then(response => response[0]);
    }

    /**
     * Simulate latency
     */
    private sleep(): Promise<void> {
        return new Promise((resolve) => {
            setTimeout(resolve, this.minDelay);
        });
    }
}
