import * as React from 'react';
import { IAComponentProps, IAComponentState, AComponent } from '../../acomponent';
import { ISearchForm } from '../../main/tripsorter';
import { AutocompleteSelectForm } from '../../elements/form/select/autocomplete/autocomplete';
import { IButtonRadioItem } from '../../elements/form/radio/button/button';

export interface ISearchPageProps extends IAComponentProps {
    /**
     * Callback for submitting search parameters
     */
    onSubmit?: (searchParams: ISearchForm) => void;
}

export interface ISearchPageState extends IAComponentState {
    /**
     * City from
     */
    departure: string;
    /**
     * City to
     */
    arrival: string;
    /**
     * Optimize search by
     */
    mode: string;
}

/**
 * Search page with form
 */
export abstract class SearchPage extends AComponent<ISearchPageProps, ISearchPageState> {
    /**
     * Reference for Departure input
     */
    protected inputDepartureRef: React.RefObject<AutocompleteSelectForm>;
    /**
     * Reference for Arrival input
     */
    protected inputArrivalRef: React.RefObject<AutocompleteSelectForm>;

    public constructor(props: ISearchPageProps) {
        super(props);
        this.inputDepartureRef = React.createRef();
        this.inputArrivalRef = React.createRef();
    }

    protected getInitState(state: ISearchPageState) {
        return Object.assign({
            mode: 'cost'
        }, state);
    }

    /**
     * Select mode
     * @param item Mode item
     */
    protected onSelectMode(item: IButtonRadioItem) {
        this.safeSetState({
            mode: (item.value || item.label)!
        });
    }

    /**
     * Select city from
     * @param departure City from value
     */
    protected onSelectDeparture(departure: string) {
        this.safeSetState({
            departure
        });
    }

    /**
     * Select city to
     * @param arrival City to value
     */
    protected onSelectArrival(arrival: string) {
        this.safeSetState({
            arrival
        });
    }

    /**
     * Trigger validators and submit form if everything is correct
     */
    protected onSubmit() {
        if (this.props.onSubmit) {
            if ([
                this.inputDepartureRef.current!.isValid(),
                this.inputArrivalRef.current!.isValid()
            ].includes(false)) {
                return;
            }
            this.props.onSubmit({
                arrival: this.state.arrival,
                departure: this.state.departure,
                mode: this.state.mode as 'cost' | 'duration'
            });
        }
    }
}

export default SearchPage;
