import * as React from 'react';
import { IAComponentProps, IAComponentState, AComponent } from '../../../acomponent';

interface IVector2 {
    x: number;
    y: number;
}
export interface IFilteredListItem {
    /**
     * Name of the item.
     */
    name: string;
    /**
     * Relevant score of the item.
     */
    score: number;
}

export interface IFilteredListProps extends IAComponentProps {
    /**
     * List of items to filtered
     */
    items: Array<IFilteredListItem>;
    /**
     * Current selected item by index
     */
    selectedIndex: number;
    /**
     * Callback for the selected item by user
     */
    onSelectItem?: (item: IFilteredListItem) => void;
    /**
     * Callback for the current highlighted item
     */
    onHighlightItem?: (itemIndex: number) => void;
    /**
     * Distance max in pixel before to cancel item selection by Touch event
     */
    touchMoveThreshold?: number;
}
export interface IFilteredListState extends IAComponentState { }

/**
 * List of filtered items
 */
export abstract class FilteredList extends AComponent<IFilteredListProps, IFilteredListState> {
    static defaultProps: Partial<IFilteredListProps> = {
        touchMoveThreshold: 10
    };

    protected selectedItem: React.RefObject<HTMLLIElement>;

    private itemReserved: IFilteredListItem | null;
    private vector: IVector2 | null;
    private touchLength: number;

    public constructor(props: IFilteredListProps) {
        super(props);

        this.selectedItem = React.createRef();
        this.itemReserved = null;
        this.vector = null;
        this.touchLength = 0;
    }

    protected getInitState(state: IFilteredListState) {
        return state;
    }

    /**
     * Always display the highlighted item from Mouse/Keyboard iteraction.
     */
    public componentDidUpdate() {
        if (this.selectedItem.current) {
            const parentNode = this.selectedItem.current.parentElement!;
            const { offsetTop, offsetHeight } = this.selectedItem.current;

            const positionTop = offsetTop - parentNode.scrollTop;
            const positionBottom = positionTop + offsetHeight;

            if (positionBottom > parentNode.offsetHeight) {
                parentNode.scrollBy(0, positionBottom - parentNode.offsetHeight);
            } else if (positionTop < 0) {
                parentNode.scrollBy(0, positionTop);
            }
        }
    }

    /**
     * Click on item to stop defaut event behavior
     * @param e Generic event
     */
    protected onClick(e: React.SyntheticEvent<HTMLLIElement>) {
        e.stopPropagation();
        e.preventDefault();
    }

    /**
     * Trigger which item was selected by user
     * @param item Selected item
     */
    protected onSelect(item: IFilteredListItem) {
        if (this.props.onSelectItem) {
            this.props.onSelectItem(item);
        }
    }

    /**
     * Reserve item to see user behavior
     * @param item Reserve item for Mobile Touch
     */
    protected onReserve(item: IFilteredListItem) {
        this.itemReserved = item;
        this.vector = null;
        this.touchLength = 0;
    }

    /**
     * Try to select if all conditions are met
     * @param item Latest item under finger
     */
    protected onTrySelect(item: IFilteredListItem) {
        if (!!this.itemReserved && this.itemReserved === item) {
            this.itemReserved = null;
            this.vector = null;
            this.touchLength = 0;
            this.onSelect(item);
        }
    }

    /**
     * Analyze move behavior
     * @param e TouchEvent
     */
    protected onTouchMove(e: React.TouchEvent<HTMLElement>) {
        const newVector: IVector2 = { x: e.targetTouches[0].screenX, y: e.targetTouches[0].screenY };
        if (!this.vector) {
            this.touchLength = 0;
            this.vector = newVector;
        } else {
            this.touchLength += Math.abs(this.vector.x - newVector.x) + Math.abs(this.vector.y - newVector.y);
            this.vector = newVector;

            if (this.touchLength > this.props.touchMoveThreshold!) {
                this.itemReserved = null;
            }
        }
    }

    /**
     * Trigger which item is currently highlighted by user
     * @param index Item's index
     */
    protected onOver(index: number) {
        if (this.props.onHighlightItem) {
            this.props.onHighlightItem(index);
        }
    }
}

export default FilteredList;
