import * as React from 'react';
import { Autocomplete } from '../../../../../services/autocomplete';
import { IFormElementProps, IFormElementState, AFormElement } from '../../aformelement';
import { IFilteredListItem } from '../../../list/fltered/filtered';

export enum ENavigationMode {
    /**
     * The navigation is circular
     */
    Circular,
    /**
     * The navigation stop on the boundary
     */
    Limit
}

interface IAutocompleteSelectFormProps extends IFormElementProps {
    /**
     * Default placeholder for the search input
     */
    placeholder: string;
    /**
     * Provide service for Autocomplete
     */
    autocompleteService?: Autocomplete;
    /**
     * Optional list of exclude value to filter after calling Autocomplete service
     */
    exclude?: Array<string>;
    /**
     * Callback on which item user select
     */
    onSelectItem?: (item: string) => void;
    /**
     * Navigation mode
     */
    navigation?: ENavigationMode;
}
interface IAutocompleteSelectFormState<I> extends IFormElementState {
    /**
     * Current value of search input
     */
    value: string;
    /**
     * Specify index of the current highlighted item in the suggestbox
     */
    highlightItemIndex: number;
    /**
     * Search input is focus
     */
    isFocus: boolean;
    /**
     * List of Suggest item
     */
    suggest: Array<I>;
    /**
     * Specify if Autocomplete service should be called
     */
    shouldReloadSuggest: boolean;
    /**
     * Previous props to compare with new props
     */
    previousProps: IAutocompleteSelectFormProps;
}

/**
 * Autocomplete input with Suggestbox.
 */
export abstract class AutocompleteSelectForm extends AFormElement<IAutocompleteSelectFormProps, IAutocompleteSelectFormState<IFilteredListItem>> {
    /**
     * Reference for Search Input
     */
    protected searchInputRef: React.RefObject<HTMLInputElement>;

    static defaultProps: Partial<IAutocompleteSelectFormProps> = {
        exclude: [],
        navigation: ENavigationMode.Circular
    };

    public constructor(props: IAutocompleteSelectFormProps) {
        super(props);

        this.searchInputRef = React.createRef<HTMLInputElement>();
    }

    protected getInitState(state: IAutocompleteSelectFormState<IFilteredListItem>): IAutocompleteSelectFormState<IFilteredListItem> {
        return Object.assign(state, {
            value: '',
            highlightItemIndex: 0,
            isFocus: false,
            suggest: [],
            shouldReloadSuggest: false,
            previousProps: this.props
        } as Partial<IAutocompleteSelectFormState<IFilteredListItem>>);
    }

    public componentDidMount(): void {
        super.componentDidMount();
        this.getSuggest(this.state.value);
    }

    public static getDerivedStateFromProps(newProps: IAutocompleteSelectFormProps, state: IAutocompleteSelectFormState<IFilteredListItem>) {
        const change: Partial<IAutocompleteSelectFormState<IFilteredListItem>> = {};
        if (newProps.exclude) {
            if (newProps.exclude.includes(state.value)) {
                change.value = '';
                change.shouldReloadSuggest = true;
            }

            change.shouldReloadSuggest = change.shouldReloadSuggest ||
                !state.previousProps.exclude ||
                newProps.exclude.length !== state.previousProps.exclude.length ||
                !!newProps.exclude.find(exclude => !state.previousProps.exclude!.find(prevExclude => prevExclude === exclude));

        }

        change.previousProps = newProps;
        return change;
    }

    /**
     * Check if the Element is valid
     */
    public isValid(): boolean {

        if (!this.searchInputRef.current!.checkValidity()) {
            this.safeSetState({
                error: this.searchInputRef.current!.validationMessage
            });
            return false;
        }

        const lowerValue = this.state.value.toLowerCase();
        if (!this.state.suggest.find(suggest => suggest.name.toLowerCase() === lowerValue)) {
            this.safeSetState({
                error: `"${this.state.value}" is not a valid entry. Please select an entry from list.`
            });
            return false;
        }

        this.safeSetState({
            error: ''
        });

        return true;
    }

    /**
     * Exclude the optional values and sort by score or name.
     * @param result Result from Autocomplete service
     */
    private filterSuggest(result: Array<IFilteredListItem>): Array<IFilteredListItem> {
        return (result.map(city => {
            if (this.props.exclude!.includes(city.name)) {
                return;
            }
            return {
                name: city.name,
                score: city.score || 0
            } as IFilteredListItem;
        }).filter(Boolean) as Array<IFilteredListItem>)
            .sort((a, b) => {
                if (a!.score !== b!.score) {
                    return b!.score - a!.score;
                }
                return a!.name.toLowerCase().localeCompare(b!.name.toLowerCase());
            });
    }

    private getSuggest(term: string): void {
        if (this.props.autocompleteService) {
            this.props.autocompleteService.find(term).then((result) => {
                if (this.state.value === term || this.state.shouldReloadSuggest) {
                    const newState: Partial<IAutocompleteSelectFormState<IFilteredListItem>> = {
                        shouldReloadSuggest: false
                    };
                    const newSuggest = this.filterSuggest(Array.from(result));

                    const hasNewSuggest = newSuggest.toString() !== this.state.suggest.toString();
                    if (hasNewSuggest) {
                        newState.suggest = newSuggest;
                    }
                    this.safeSetState(() => newState as IAutocompleteSelectFormState<IFilteredListItem>, () => {
                        if (hasNewSuggest) {
                            this.onHightlightItem(0);
                        }
                    });
                }
            });
        }
    }

    /**
     * Callback if user write something in the search input
     */
    protected onChange(e: React.ChangeEvent<HTMLInputElement>) {
        this.safeSetState({
            value: e.currentTarget.value,
            error: ''
        }, () => {
            this.getSuggest(this.state.value);
        });
    }

    /**
     * Callback for selected suggested item. Reload the full suggestbox.
     * @param item Selected item
     */
    protected onSelectItem(item: IFilteredListItem) {
        this.safeSetState({
            value: item.name,
            error: ''
        }, () => {
            this.getSuggest('');
            this.searchInputRef.current!.blur();
            if (this.props.onSelectItem) {
                this.props.onSelectItem(this.state.value);
            }
        });
    }

    /**
     * Set the index of the highlighted item.
     * @param highlightItemIndex Index of the highlighted item
     */
    protected onHightlightItem(highlightItemIndex: number) {
        this.safeSetState({
            highlightItemIndex
        });
    }

    /**
     * Trigger search input focus
     */
    protected onInputFocus() {
        this.safeSetState({
            isFocus: true
        });
    }

    /**
     * Trigger search input blur
     */
    protected onInputBlur() {
        this.safeSetState({
            isFocus: false
        });
    }

    private applyCircularNavigation(offset: number) {
        let index = this.state.highlightItemIndex + offset;
        if (offset > 0) {
            if (index >= this.state.suggest.length) {
                index = 0;
            }
        } else {
            if (index < 0) {
                index = this.state.suggest.length - 1;
            }
        }
        return index;
    }

    private applyLimitNavigation(direction: 'ArrowDown' | 'ArrowUp') {
        const [minMax, limit, offset] = direction === 'ArrowDown' ?
            [Math.min, this.state.suggest.length - 1, 1] :
            [Math.max, 0, -1];
        return minMax(this.state.highlightItemIndex + offset, limit);
    }

    /**
     * Catch all keyboard event to allow navigation on the suggestbox
     * @param e Key event
     */
    protected onInputKeyDown(e: React.KeyboardEvent<HTMLInputElement>) {
        if (e.key === 'ArrowUp' || e.key === 'ArrowDown') {
            e.preventDefault();
            let highlightItemIndex: number;
            if (this.props.navigation === ENavigationMode.Circular) {
                highlightItemIndex = this.applyCircularNavigation(e.key === 'ArrowDown' ? 1 : -1);
            } else {
                highlightItemIndex = this.applyLimitNavigation(e.key);
            }
            this.safeSetState({ highlightItemIndex });
        } else if (e.key === 'Enter' || e.key === 'Tab') {
            if (e.key === 'Enter') {
                e.preventDefault();
            }
            if (e.shiftKey === false || e.key === 'Enter') {
                this.onSelectItem(this.state.suggest[this.state.highlightItemIndex]);
            }
        }
    }

    /**
     * Reload suggestbox
     */
    protected prepareReloadSuggest() {
        const isValidSuggest = !!this.state.suggest.find(suggest => suggest.name === this.state.value);
        this.getSuggest(isValidSuggest ? '' : this.state.value);
    }

    public abstract render(): JSX.Element;
}

export default AutocompleteSelectForm;
