import * as React from 'react';

const typeOfString = typeof '';
function reactcomponent_applyToChildren(children: Array<JSX.Element|string>, attribute: string, componentName: string) {
    if (!(children instanceof Array)) {
        return children;
    }
    const newChildren = children.map(child => {
        if (typeof child === typeOfString || typeof (child as JSX.Element).type !== typeOfString) {
            return child;
        }

        return reactcomponent_applyToElement(child as JSX.Element, attribute, componentName, true);
    });

    if (Object.freeze) {
        return Object.freeze(newChildren);
    }
    return newChildren;
}

function reactcomponent_applyToElement(element: JSX.Element, attribute: string, componentName: string, recursive: boolean) {
    const children: any = recursive ? reactcomponent_applyToChildren(element.props.children, attribute, componentName) : element.props.children;
    const props = Object.assign({
        [attribute]: componentName
    }, element.props);
    ['children'].forEach(attr => delete props[attr]);

    return React.createElement(element.type, props, children);
}

/**
 * Add custom attribute with the name of the React Component.
 * If the default constructor name is used, take care about Minifier that you should keep function/constructor name.
 * @example UglifyJS: uglifyjs main.js --keep-fnames -o build.js
 * @param recursive If true, go throught all children to apply react component name
 * @param attribute Name of the custom attribute. This version will not merge existing attribute value.
 * @param name Value of the attribute. Default: name of the constructor.
 */
export function reactcomponent(recursive: boolean = false, attribute: string = 'component', name?: string) {
    return function custom(target: any, method: string, descriptor: PropertyDescriptor) {

        if (method !== 'render') {
            return;
        }

        const originalMethod = target[method];
        descriptor.value = target[method] = function() {
            const renderedElement: JSX.Element = originalMethod.call(this);

            return reactcomponent_applyToElement(renderedElement, attribute, name || target.constructor.name, recursive);
        };
    };
}
